import torch
from transformers import AutoTokenizer, AutoModelForCausalLM, TrainingArguments, Trainer
from peft import LoraConfig, get_peft_model

model_name = "Qwen/Qwen2.5-7B-Instruct"
# 1) 加载分词器
# [   26   ]
# 下面的模型加载时使用了trust_remote_code=True，题目中也提到了要求与模型端一致的remote_code
# 因此这个题目选择 D, 选项A中的return_tensorts="pt" 是tokenizer()中的参数，意思是返回Pytorch格式张量
tokenizer = AutoTokenizer.from_pretrained(model_name,trust_remote_code=True)
tokenizer=tokenizer(return_tensors="pt",)
model = AutoModelForCausalLM.from_pretrained(
    model_name, device_map="auto", torch_dtype=torch.float16,
    trust_remote_code=True
    # device_map="auto" 自动将模型层分配到可用设备（如多GPU）
    # torch_dtype=torch.float16 使用半精度（FP16）加载权重，节省显存
    # trust_remote_code=True  允许运行模型仓库中的自定义代码（Qwen 需要）
)

# 2) 配置 Lora
lora_config = LoraConfig(
    #[   27  ]
    # LoRA 秩参数 r 的取值 一般为 8~128，具体大小视任务要求设定
    # 小规模任务：8-32
    # 中等规模任务：32-64
    # 大规模任务：64-128
    # 复杂任务：128-256

    # lora_alpha 是缩放因子，缩放因子控制LoRA适配器对原始权重的贡献程度
    # lora_alpha 类似于学习率的作用。通常设置为秩的2倍或4倍。

    # 本题中符合常见lora_alpha和r比例关系的仅有C选项，若存在选项 r=16 lora_alpha=32
    # 也应选择 r=8，lora_alpha=16，因为题目强调了资源受限与节省现存，要选择更小的 r
    r=8, lora_alpha=16,
    #[   28  ]
    # 有关target_modules的通常取值有以下取值
    # 1.注意力模块:[“q_proj”, “k_proj”, “v_proj”, “o_proj”]
    # q_proj 查询投影层 作用:决定了每个 token 如何去查询其他 token
    # k_proj 键投影层  作用: 键更像是静态索引，变化较小
    # v_proj 值投影层  作用: 值是实际要提取的信息内容。
    # o_proj 输出投影层 作用: 注意力机制的最终线性变换
    # 2.前馈网络(FNN)模块:[“gate_proj”, “up_proj”, “down_proj”]
    # gate_proj 门控投影层 作用：生成一个“门控信号”，决定多少信息可以通过。
    # up_proj 上投影层  作用：将输入映射到高维空间，生成需要被“门控”的值。
    # down_proj 下投影层 作用：将门控后的高维表示重新投影回原始维度（通常是隐藏层大小）。
    # 3.全模块

    # 在大量 LoRA 微调实验中发现，将适配器（LoRA）应用于 q_proj 和 v_proj 层，往往就能达到接近全参数微调的效果，且训练资源消耗少。
    # 因此这个题选择 ["q_proj","v_proj"]，但并不代表其他的模块没有作用，在LoRA微调中，如果性能不足，逐步添加其他模块
    # 在 LoRA 微调中，q_proj 和 v_proj 是最常被选择的注意力模块，因为它们对注意力模式和语义表达的影响最大，能在较小代价下获得良好性能
    target_modules=["q_proj","v_proj"],
    lora_dropout=0.05, # 在 LoRA 适配器中加入 dropout，防止过拟合，重要参数
    #主流框架默认值：#Hugging Face PEFT：lora_dropout=0.05
    #QLoRA 论文实验：多使用 0.05 或 0.1
    # lora_dropout=0.0	 大数据集（> 10K 样本）	不启用 dropout，训练更快
    # lora_dropout=0.05	 最常用默认值	平衡正则化与性能，Hugging Face PEFT 默认
    # lora_dropout=0.1	 中小数据集（1K ~ 5K）	增强正则化，防过拟合
    # lora_dropout=0.2~0.3	极小数据集（< 1K）	强正则化，但可能欠拟合
    bias="none",  # 不训练偏置项（更高效）
    task_type="CAUSAL_LM"  # 任务类型：因果语言建模（自回归生成）
)
# 将 LoRA 适配器注入原始模型，返回一个可训练的 PEFT 模型
model = get_peft_model(model, lora_config)

# 占位：训练数据集（实际使用时需替换为真实 dataset） 这边时假设已经准备好
train_dataset=0

# 3) 定义 Trainer (假设 train_dataset 已经准备好)
training_args = TrainingArguments(
    output_dir="./lora-qwen2.5-7b",# 训练输出目录（保存 checkpoint）
    #[  29  ]
    # per_device_train_batch_size 每个 GPU 设备上单次前向/反向传播所使用的样本数量。
    # gradient_accumulation_steps  在更新模型参数之前，先累积 多少个 batch 的梯度。
    # num_devices GPU数量
    # 计算公式: 全局batch_size = per_device_train_batch_size × gradient_accumulation_steps × num_devices

    # 本题中要求全局 batch_size 的大小为 8，同时本题中给出的资源是单卡，符合要求的有 A D 三个选项
    # 但要考虑显存限制和内存溢出的风险，那此时per_device_train_batch_size要尽可能小，因此选 D
    # 对于 7B 以上的大模型，per_device_train_batch_size=1 是常见甚至必要的选择，因为更大的 batch size 可能导致 OOM（显存溢出）
    per_device_train_batch_size=1,gradient_accumulation_steps=8,
    num_train_epochs=3, # 总共训练 3 轮
    learning_rate=2e-4, # 学习率 关键超参数，直接影响训练的稳定性、收敛速度和最终性能。常用值  1e-4 ~ 5e-4
    # 通用 LoRA 微调（7B) 2e-4
    #[  30  ]
    # fp32（单精度）  全精度参数；占用4字节                           全量微调默认
    # fp16 (半精度)  半精度参数 ；占用16bits                        显存紧张，需配合损失缩放
    # bf16 (brain float point 脑浮点)  半精度参数 ；占用2字节        推荐用于大模型（需硬件支持）
    # 在当前（2025年）的大模型训练中，BF16 已成为主流和强烈推荐的精度格式，
    # 尤其是在使用现代硬件（如 NVIDIA A100、H100、RTX 30/40 系列）进行大规模语言模型（LLM）训练或微调时。
    # 本题虽然没有给出显卡型号，但模型加载时设置了 torch_dtype=torch.float16 ，因此选择使用fp16，选择B，C和D选项使用错误
    fp16=True,
    logging_steps=10, # 每 10 步记录一次日志
    save_strategy="epoch"  # 每个 epoch 结束后保存一次 checkpoint
)
# 初始化 Hugging Face Trainer，封装训练逻辑
trainer = Trainer(
    model=model,# 带 LoRA 适配器的模型
    args=training_args,# 训练参数配置
    train_dataset=train_dataset, # 训练数据集
    tokenizer=tokenizer, # 分词器
)
# 开始训练（执行梯度更新、loss 记录等）
trainer.train()

# 4) 保存 Lora 权重
model.save_pretrained("./lora-qwen2.5-7b-lora")


